<?php
session_start();

// Require login
if (!isset($_SESSION['user_id'])) {
    header("Location: login.php");
    exit;
}

// If role is not admin, send to view page (view-only)
if (empty($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    // optional friendly message via query param
    header("Location: view.php?msg=forbidden");
    exit;
}

/* DB */
$mysqli = new mysqli("localhost","root","","powerbi_demo");
if ($mysqli->connect_errno) die("DB error: " . $mysqli->connect_error);

/* Fetch all users except current (to assign file to someone else) */
$stmt = $mysqli->prepare("SELECT id, username FROM users WHERE id != ? ORDER BY username ASC");
$stmt->bind_param("i", $_SESSION['user_id']);
$stmt->execute();
$users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt->close();

$success = $error = null;
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (empty($_POST['assigned_user'])) { $error = "Please select a user to assign."; }
    elseif (!isset($_FILES['file'])) { $error = "Please choose an HTML file to upload."; }
    else {
        $assigned_user = (int)$_POST['assigned_user'];
        $file = $_FILES['file'];
        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        // Basic checks
        if ($file['error'] !== UPLOAD_ERR_OK) {
            $error = "Upload failed (code {$file['error']}).";
        } elseif ($ext !== 'html' && $ext !== 'htm') {
            $error = "Only .html files allowed.";
        } else {
            // sanitize original filename part
            $safe_base = preg_replace('/[^a-z0-9\-_\.]/i','_',pathinfo($file['name'], PATHINFO_FILENAME));
            $newname = time() . "_" . bin2hex(random_bytes(4)) . "_" . $safe_base . ".html";
            $dest = __DIR__ . '/uploads/' . $newname;

            if (!is_dir(__DIR__ . '/uploads')) mkdir(__DIR__ . '/uploads', 0755, true);

            if (move_uploaded_file($file['tmp_name'], $dest)) {
                $stmt = $mysqli->prepare("INSERT INTO assign_user (uploader_id, assigned_user_id, file_name) VALUES (?, ?, ?)");
                $stmt->bind_param("iis", $_SESSION['user_id'], $assigned_user, $newname);
                if ($stmt->execute()) {
                    $success = "File uploaded and assigned to user successfully.";
                } else {
                    $error = "DB error saving assignment.";
                    @unlink($dest);
                }
            } else {
                $error = "Failed to move uploaded file.";
            }
        }
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Upload — DustValue Report (Admin)</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    body { background: linear-gradient(180deg,#f7fbff,#f0f7ff); font-family:Inter,system-ui,-apple-system; padding:40px;}
    .panel { max-width:980px; margin:auto; }
    .card-creative { border-radius:16px; box-shadow: 0 10px 30px rgba(15,23,42,.06); }
    .dropzone {
      border: 2px dashed rgba(34,156,255,.14);
      border-radius:12px; padding:28px; text-align:center; transition:all .2s;
      background: linear-gradient(180deg, rgba(250,250,255,.8), rgba(245,248,255, .6));
      cursor:pointer;
    }
    .dropzone.drag { box-shadow: 0 10px 30px rgba(34,156,255,.08); transform:translateY(-3px); }
    .tiny { font-size:.9rem; color:#6b7280; }
    .topbar { display:flex; justify-content:space-between; align-items:center; margin-bottom:18px; }
  </style>
</head>
<body>
<div class="panel">
  <div class="topbar">
    <h3>Upload & Assign <small class="text-muted">(.html only) — Admin</small></h3>
    <div>
      <a class="btn btn-outline-danger" href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
    </div>
  </div>

  <div class="card card-creative p-4 mb-4">
    <?php if ($success): ?>
      <div class="alert alert-success"><?=htmlspecialchars($success)?></div>
    <?php elseif ($error): ?>
      <div class="alert alert-danger"><?=htmlspecialchars($error)?></div>
    <?php endif; ?>

    <form method="post" enctype="multipart/form-data" id="uploadForm">
      <div class="row g-3">
        <div class="col-lg-5">
          <label class="form-label">Assign to user</label>
          <select name="assigned_user" class="form-select" required>
            <option value="">— select user —</option>
            <?php foreach($users as $u): ?>
              <option value="<?= (int)$u['id'] ?>"><?= htmlspecialchars($u['username']) ?></option>
            <?php endforeach; ?>
          </select>

          <div class="mt-3 tiny">Tip: The dropdown excludes the user who is currently logged in (you).</div>

          <div class="mt-4">
            <label class="form-label">Upload .html file</label>
            <div id="dz" class="dropzone" onclick="document.querySelector('#fileInput').click()">
              <div><i class="bi bi-file-earmark-code" style="font-size:36px;color:#4f46e5"></i></div>
              <div style="font-weight:600;margin-top:8px">Click to select file or drag & drop</div>
              <div class="tiny mt-2">Only .html / .htm allowed. Max set by server (php.ini).</div>
              <input type="file" id="fileInput" name="file" accept=".html,.htm" style="display:none">
            </div>
            <div id="selected" class="mt-2"></div>
          </div>
        </div>

        <div class="col-lg-7">
          <label class="form-label">Preview (server files assigned to others)</label>
          <div class="p-3" style="background:#fff;border-radius:12px;min-height:220px;border:1px solid #eef2ff">
            <div class="text-muted">No preview for new file until uploaded. Existing files assigned by you appear in the Viewer page.</div>
          </div>
        </div>
      </div>
    </form>
  </div>

  <div class="text-center text-muted small">Uploads are saved under <code>/uploads/</code></div>
</div>

<script>
const dz = document.getElementById('dz');
const fileInput = document.getElementById('fileInput');
const selected = document.getElementById('selected');

dz.addEventListener('dragover', e => { e.preventDefault(); dz.classList.add('drag'); });
dz.addEventListener('dragleave', e => { dz.classList.remove('drag'); });
dz.addEventListener('drop', e => {
  e.preventDefault(); dz.classList.remove('drag');
  if (e.dataTransfer.files && e.dataTransfer.files.length) {
    fileInput.files = e.dataTransfer.files;
    showSelected();
  }
});

fileInput.addEventListener('change', showSelected);
function showSelected(){
  if (!fileInput.files.length) { selected.innerHTML = ''; return; }
  const f = fileInput.files[0];
  selected.innerHTML = `<div class="mt-2"><strong>${f.name}</strong> — ${Math.round(f.size/1024)} KB</div>`;
}
</script>
</body>
</html>
