<?php
session_start();
if (!isset($_SESSION['user_id'])) { header("Location: login.php"); exit; }

$mysqli = new mysqli("localhost","root","","powerbi_demo");
if ($mysqli->connect_errno) die("DB error: " . $mysqli->connect_error);

$userId = (int)$_SESSION['user_id'];
$userRole = $_SESSION['role'] ?? 'user'; // 'admin' or 'user'

/* --- Helpers --- */
function h($v) { return htmlspecialchars((string)($v ?? ''), ENT_QUOTES, 'UTF-8'); }
function fmt_date($v, $format = 'd M Y H:i') {
    if (empty($v)) return '';
    $ts = strtotime($v);
    if ($ts === false) return '';
    return date($format, $ts);
}
function get_base_url() {
    $https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || ($_SERVER['SERVER_PORT'] ?? '') === '443';
    $scheme = $https ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $scriptDir = rtrim(str_replace('\\','/', dirname($_SERVER['PHP_SELF'])), '/');
    if ($scriptDir === '') $scriptDir = '/';
    return $scheme . '://' . $host . $scriptDir;
}
function proxy_url($id, $action = '') {
    $id = (int)$id;
    $base = rtrim(get_base_url(), '/');
    $url = $base . '/download.php?id=' . rawurlencode($id);
    if ($action === 'download') $url .= '&action=download';
    return $url;
}
function direct_file_url_by_filename($filename) {
    $base = rtrim(get_base_url(), '/');
    return $base . '/uploads/' . rawurlencode($filename);
}

/* --- Fetch files depending on role --- */
if ($userRole === 'admin') {
    // Admin sees all files; include uploader + assigned user
    $sql = "
      SELECT a.id, a.file_name, a.uploaded_at,
             u.username AS uploader,
             au.username AS assigned_username,
             a.assigned_user_id
      FROM assign_user a
      JOIN users u ON u.id = a.uploader_id
      LEFT JOIN users au ON au.id = a.assigned_user_id
      ORDER BY a.uploaded_at DESC
    ";
    $res = $mysqli->query($sql);
    $files = $res ? $res->fetch_all(MYSQLI_ASSOC) : [];
} else {
    // regular user: only files assigned to them
    $stmt = $mysqli->prepare("
      SELECT a.id, a.file_name, a.uploaded_at, u.username AS uploader
      FROM assign_user a
      JOIN users u ON u.id = a.uploader_id
      WHERE a.assigned_user_id = ?
      ORDER BY a.uploaded_at DESC
    ");
    $stmt->bind_param("i", $userId);
    $stmt->execute();
    $files = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}

/* --- Determine selected file (GET file_id) with authorization --- */
$selected = null;
$errorMsg = null;
if (!empty($_GET['file_id'])) {
    $fid = (int)$_GET['file_id'];

    if ($userRole === 'admin') {
        // admin can view any file
        $q = $mysqli->prepare("SELECT a.id, a.file_name, a.uploaded_at, a.assigned_user_id, u.username AS uploader FROM assign_user a JOIN users u ON u.id = a.uploader_id WHERE a.id = ? LIMIT 1");
        $q->bind_param("i", $fid);
    } else {
        // user must be assigned
        $q = $mysqli->prepare("SELECT a.id, a.file_name, a.uploaded_at, a.assigned_user_id, u.username AS uploader FROM assign_user a JOIN users u ON u.id = a.uploader_id WHERE a.id = ? AND a.assigned_user_id = ? LIMIT 1");
        $q->bind_param("ii", $fid, $userId);
    }
    $q->execute();
    $res = $q->get_result()->fetch_assoc();
    $q->close();

    if ($res) {
        $selected = $res;
    } else {
        $errorMsg = "The selected file is not available to you or does not exist.";
    }
}

// default select first
if (!$selected && !empty($files)) $selected = $files[0];

/* --- Build iframe src if file exists --- */
$iframeSrc = '';
if ($selected) {
    $path = __DIR__ . '/uploads/' . ($selected['file_name'] ?? '');
    if (file_exists($path) && is_readable($path)) {
        $iframeSrc = direct_file_url_by_filename($selected['file_name']);
    } else {
        $errorMsg = $errorMsg ?: "File missing on server.";
    }
}
?>
<!doctype html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>Files — Viewer</title>
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
  <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet">
  <style>
    :root{ --accent1: #7b2ff7; --accent2: #00d4ff; }
    html,body { height:100%; margin:0; font-family:Inter,system-ui,-apple-system; background:#0b1220; color:#e6eef8; }
    .topbar {
      position:fixed; left:0; right:0; top:0; height:64px;
      display:flex; align-items:center; justify-content:space-between;
      padding:0 18px; box-shadow:0 8px 30px rgba(2,6,23,0.6);
      background: linear-gradient(90deg,var(--accent1),var(--accent2));
      z-index:1000;
    }
    .brand { display:flex; align-items:center; gap:12px; color:white; }
    .logo { width:40px;height:40px;border-radius:8px;background:white;color:var(--accent1);display:grid;place-items:center;font-weight:800; }
    .top-actions { display:flex; gap:8px; align-items:center; }
    .file-card {
      position: fixed; left: 18px; top: 86px; z-index: 900;
      width: 360px; background: rgba(255,255,255,0.04);
      border: 1px solid rgba(255,255,255,0.06); border-radius: 10px; padding: 10px;
      box-shadow: 0 10px 30px rgba(2,6,23,0.6);
    }
    .file-card label { color:#cfe8ff; font-size:.9rem; display:block; margin-bottom:6px; }
    .file-card .form-select { background: rgba(255,255,255,0.02); color:#e6eef8; border:1px solid rgba(255,255,255,0.06); }
    .preview-wrap { position:fixed; left:0; right:0; top:64px; bottom:0; display:flex; align-items:center; justify-content:center; background:linear-gradient(180deg, rgba(2,6,23,0.6), rgba(2,6,23,0.85)); }
    .preview-frame { width:100%; height:calc(100vh - 74px); border:0; box-shadow: 0 20px 50px rgba(2,6,23,0.6); background:white; }
    .floating-meta { position:fixed; right:18px; top:86px; z-index:910; color:#cfe8ff88; font-size:.9rem; }
    .no-file { padding:60px; text-align:center; color:#94a3b8; background:white; border-radius:12px; width:80%; margin:0 auto; color:#1f2937; }
    .debug { font-size:.85rem; color:#cfe8ff88; margin-top:8px; word-break:break-all; }
    a.btn-ghost { background: rgba(255,255,255,0.08); color: #fff; border-radius:8px; padding:8px 12px; text-decoration:none; }
  </style>
</head>
<body>
  <div class="topbar">
    <div class="brand">
      <div class="logo">DVR</div>
      <div>
        <div style="font-weight:700">DustValue Report Viewer</div>
        <div style="font-size:.85rem; opacity:.9"><?= h($_SESSION['username'] ?? '') ?> <small style="opacity:.8; margin-left:6px; font-size:.8rem;"><?= h($userRole) ?></small></div>
      </div>
    </div>

    <div class="top-actions">
      <?php if ($userRole === 'admin'): ?>
        <a class="btn-ghost" href="upload.php"><i class="bi bi-cloud-upload"></i> Upload</a>
        <a class="btn-ghost" href="users_list.php"><i class="bi bi-people"></i> Users</a>
      <?php endif; ?>
      <a class="btn-ghost" href="logout.php"><i class="bi bi-box-arrow-right"></i> Logout</a>
    </div>
  </div>

  <!-- file dropdown card top-left -->
  <div class="file-card" role="region" aria-label="Select file to preview">
    <form id="fileForm" method="get">
      <label for="file_id">Select file</label>
      <select id="file_id" name="file_id" class="form-select" onchange="document.getElementById('fileForm').submit();">
        <?php if (empty($files)): ?>
          <option value="">— no files available —</option>
        <?php else: foreach($files as $f): $fid = (int)$f['id']; ?>
          <option value="<?= $fid ?>" <?= ($selected && (int)($selected['id'] ?? 0) === $fid) ? 'selected' : '' ?>>
            <?= h($f['file_name']) ?>
            <?php if (isset($f['assigned_username'])): ?> — assigned: <?= h($f['assigned_username']) ?><?php endif; ?>
            — uploader: <?= h($f['uploader'] ?? '') ?>
          </option>
        <?php endforeach; endif; ?>
      </select>

      <?php if ($selected): ?>
        <div style="margin-top:8px; color:#cfe8ffcc; font-size:.9rem;">
          <div><strong><?= h($selected['file_name'] ?? '') ?></strong></div>
          <div class="debug">Uploaded: <?= h(fmt_date($selected['uploaded_at'] ?? '')) ?></div>
        </div>

        <div style="margin-top:10px; display:flex; gap:8px;">
          <a class="btn btn-sm btn-light" href="<?= h(direct_file_url_by_filename($selected['file_name'])) ?>" target="_blank"><i class="bi bi-box-arrow-up-right"></i> Open</a>
          <a class="btn btn-sm btn-secondary" href="<?= h(proxy_url($selected['id'] ?? 0, 'download')) ?>"><i class="bi bi-download"></i> Download</a>
        </div>
      <?php endif; ?>
    </form>
  </div>

  <!-- floating meta (right) -->
  <div class="floating-meta">
    <?php if (!empty($errorMsg)): ?>
      <div style="background:rgba(255,255,255,0.06); padding:8px 10px; border-radius:8px; color:#ffd7d7;"><?= h($errorMsg) ?></div>
    <?php endif; ?>
  </div>

  <!-- full screen preview -->
  <div class="preview-wrap" id="previewWrap">
    <?php if ($iframeSrc): ?>
      <iframe id="previewFrame" class="preview-frame" sandbox="allow-scripts allow-same-origin allow-forms allow-popups" src="<?= h($iframeSrc) ?>" title="<?= h($selected['file_name'] ?? 'Preview') ?>"></iframe>
    <?php else: ?>
      <div class="no-file">
        <div style="font-size:20px; font-weight:700; color:#0f172a;">Nothing to preview</div>
        <div style="margin-top:8px; color:#334155;">Select a file from the top-left to preview it here.</div>
      </div>
    <?php endif; ?>
  </div>

  <script>
    // optional: keyboard shortcuts: n = next file, p = previous file
    (function(){
      const select = document.getElementById('file_id');
      if (!select) return;
      document.addEventListener('keydown', function(e){
        if (e.target && (e.target.tagName==='INPUT' || e.target.tagName==='TEXTAREA')) return;
        if (e.key === 'n') {
          if (select.selectedIndex < select.options.length - 1) { select.selectedIndex++; select.form.submit(); }
        } else if (e.key === 'p') {
          if (select.selectedIndex > 0) { select.selectedIndex--; select.form.submit(); }
        }
      });
    })();
  </script>
</body>
</html>
